<?php
// Security first - prevent direct access
if (!defined('IN_BOT')) {
    die('Direct access not permitted');
}

require_once 'config.php';

/**
 * Database connection class using PDO
 * Provides singleton pattern to prevent multiple connections
 */
class Database {
    private static $instance = null;
    private $connection = null;
    private $error = '';

    /**
     * Private constructor to prevent direct instantiation
     */
    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false
            ];

            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);
            $this->connection->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch (PDOException $e) {
            $this->error = $e->getMessage();
            if (defined('DEBUG_MODE') && DEBUG_MODE) {
                throw new Exception("Database connection error: " . $this->error);
            } else {
                throw new Exception("Database connection failed. Please contact support.");
            }
        }
    }

    /**
     * Get singleton instance
     * @return Database
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Get PDO connection
     * @return PDO
     */
    public function getConnection() {
        return $this->connection;
    }

    /**
     * Get last error
     * @return string
     */
    public function getError() {
        return $this->error;
    }

    /**
     * Execute a query
     * @param string $sql
     * @param array $params
     * @return PDOStatement
     */
    public function query($sql, $params = []) {
        try {
            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            $this->error = $e->getMessage();
            if (defined('DEBUG_MODE') && DEBUG_MODE) {
                throw new Exception("Query error: " . $this->error);
            } else {
                throw new Exception("Database query failed. Please try again.");
            }
        }
    }

    /**
     * Begin transaction
     * @return bool
     */
    public function beginTransaction() {
        return $this->connection->beginTransaction();
    }

    /**
     * Commit transaction
     * @return bool
     */
    public function commit() {
        return $this->connection->commit();
    }

    /**
     * Rollback transaction
     * @return bool
     */
    public function rollback() {
        return $this->connection->rollBack();
    }

    /**
     * Prevent cloning
     */
    private function __clone() {}

    /**
     * Prevent unserialization
     */
    private function __wakeup() {}
}

/**
 * Initialize database
 * @return bool
 */
function initDatabase() {
    try {
        $db = Database::getInstance();
        
        // Create users table if not exists
        $sql = "CREATE TABLE IF NOT EXISTS users (
            telegram_id BIGINT PRIMARY KEY,
            username VARCHAR(255),
            balance DECIMAL(10,2) DEFAULT 0,
            income DECIMAL(10,2) DEFAULT 0,
            total_clicks BIGINT DEFAULT 0,
            referral_id BIGINT NULL,
            referral_count INT DEFAULT 0,
            last_click TIMESTAMP,
            last_claim TIMESTAMP,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB";
        
        $db->query($sql);
        
        // Create animals table if not exists
        $sql = "CREATE TABLE IF NOT EXISTS animals (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(50) NOT NULL,
            cost DECIMAL(10,2) NOT NULL,
            income_per_hour DECIMAL(10,2) NOT NULL,
            image_url VARCHAR(255),
            description TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB";
        
        $db->query($sql);
        
        // Create user_animals table if not exists
        $sql = "CREATE TABLE IF NOT EXISTS user_animals (
            id BIGINT AUTO_INCREMENT PRIMARY KEY,
            user_id BIGINT NOT NULL,
            animal_id INT NOT NULL,
            quantity INT DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(telegram_id) ON DELETE CASCADE,
            FOREIGN KEY (animal_id) REFERENCES animals(id) ON DELETE CASCADE
        ) ENGINE=InnoDB";
        
        $db->query($sql);
        
        // Create referrals table if not exists
        $sql = "CREATE TABLE IF NOT EXISTS referrals (
            id BIGINT AUTO_INCREMENT PRIMARY KEY,
            referrer_id BIGINT NOT NULL,
            referred_id BIGINT NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (referrer_id) REFERENCES users(telegram_id),
            FOREIGN KEY (referred_id) REFERENCES users(telegram_id)
        ) ENGINE=InnoDB";
        
        $db->query($sql);
        
        // Create withdrawals table if not exists
        $sql = "CREATE TABLE IF NOT EXISTS withdrawals (
            id BIGINT AUTO_INCREMENT PRIMARY KEY,
            user_id BIGINT NOT NULL,
            amount DECIMAL(10,2) NOT NULL,
            payment_method VARCHAR(50) NOT NULL,
            payment_address VARCHAR(255) NOT NULL,
            status ENUM('pending', 'completed', 'rejected') DEFAULT 'pending',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(telegram_id)
        ) ENGINE=InnoDB";
        
        $db->query($sql);
        
        // Create indexes
        $sql = "CREATE INDEX IF NOT EXISTS idx_username ON users(username)";
        $db->query($sql);
        
        $sql = "CREATE INDEX IF NOT EXISTS idx_referral_id ON users(referral_id)";
        $db->query($sql);
        
        $sql = "CREATE INDEX IF NOT EXISTS idx_last_click ON users(last_click)";
        $db->query($sql);
        
        $sql = "CREATE INDEX IF NOT EXISTS idx_last_claim ON users(last_claim)";
        $db->query($sql);
        
        $sql = "CREATE INDEX IF NOT EXISTS idx_user_animals_user ON user_animals(user_id)";
        $db->query($sql);
        
        $sql = "CREATE INDEX IF NOT EXISTS idx_user_animals_animal ON user_animals(animal_id)";
        $db->query($sql);
        
        $sql = "CREATE INDEX IF NOT EXISTS idx_referrals_referrer ON referrals(referrer_id)";
        $db->query($sql);
        
        $sql = "CREATE INDEX IF NOT EXISTS idx_referrals_referred ON referrals(referred_id)";
        $db->query($sql);
        
        $sql = "CREATE INDEX IF NOT EXISTS idx_withdrawals_user ON withdrawals(user_id)";
        $db->query($sql);
        
        $sql = "CREATE INDEX IF NOT EXISTS idx_withdrawals_status ON withdrawals(status)";
        $db->query($sql);
        
        return true;
    } catch (Exception $e) {
        error_log("Database initialization error: " . $e->getMessage());
        return false;
    }
}
?>
