<?php
/**
 * Telegram Bot Web Admin Panel - Send Message
 */

require_once '../config.php';
require_once '../bot_functions.php';
require_once 'auth_check.php';

$pdo = getDBConnection();
$categories = getCategories();
$success = $error = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Handle message sending
    $title = $_POST['title'] ?? '';
    $description = $_POST['description'] ?? '';
    $category_id = $_POST['category_id'] ?? null;
    $button_text = $_POST['button_text'] ?? '';
    $button_url = $_POST['button_url'] ?? '';
    
    // Handle image upload
    $image_url = null;
    if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
        $filename = $_FILES['image']['name'];
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        if (in_array($ext, $allowed)) {
            $new_name = uniqid('img_') . '.' . $ext;
            $upload_path = dirname(__DIR__) . '/' . UPLOAD_DIR . $new_name;
            
            if (move_uploaded_file($_FILES['image']['tmp_name'], $upload_path)) {
                $image_url = UPLOAD_DIR . $new_name;
            } else {
                $error = "Failed to upload image";
            }
        } else {
            $error = "Invalid image format. Allowed: jpg, jpeg, png, gif";
        }
    }
    
    if (!$error) {
        try {
            // Save message
            $stmt = $pdo->prepare("
                INSERT INTO messages (
                    title, description, image_url, 
                    button_text, button_url, category_id,
                    created_by, sent_from
                ) VALUES (?, ?, ?, ?, ?, ?, ?, 'web')
            ");
            
            $stmt->execute([
                $title,
                $description,
                $image_url,
                $button_text ?: null,
                $button_url ?: null,
                $category_id ?: null,
                $_SESSION['admin_username']
            ]);
            
            $message_id = $pdo->lastInsertId();
            
            // Send message
            $message_data = [
                'id' => $message_id,
                'title' => $title,
                'description' => $description,
                'image_url' => $image_url,
                'button_text' => $button_text,
                'button_url' => $button_url
            ];
            
            if ($category_id) {
                $result = sendToCategory($category_id, $message_data);
            } else {
                $result = sendToAllUsers($message_data);
            }
            
            $success = "Message sent successfully! Delivered: {$result['success']}, Failed: {$result['failed']}";
        } catch (Exception $e) {
            $error = "Failed to send message: " . $e->getMessage();
            error_log("Message send error: " . $e->getMessage());
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bot Admin Panel - Send Message</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        .sidebar {
            position: fixed;
            top: 0;
            bottom: 0;
            left: 0;
            z-index: 100;
            padding: 48px 0 0;
            box-shadow: inset -1px 0 0 rgba(0, 0, 0, .1);
            background: #f8f9fa;
        }
        .sidebar-sticky {
            position: relative;
            top: 0;
            height: calc(100vh - 48px);
            padding-top: .5rem;
            overflow-x: hidden;
            overflow-y: auto;
        }
        .main {
            padding-top: 48px;
        }
        #message-preview {
            background: #fff;
            border: 1px solid #dee2e6;
            border-radius: 10px;
            padding: 20px;
            max-width: 400px;
            margin: 20px auto;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif;
        }
        .preview-image {
            width: 100%;
            max-height: 200px;
            object-fit: cover;
            border-radius: 5px;
            margin-bottom: 10px;
        }
        .preview-button {
            display: inline-block;
            padding: 5px 15px;
            background: #0088cc;
            color: white;
            text-decoration: none;
            border-radius: 3px;
            margin-top: 10px;
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-dark bg-primary fixed-top">
        <div class="container-fluid">
            <a class="navbar-brand" href="#">📱 Bot Admin Panel</a>
            <div class="d-flex">
                <span class="navbar-text me-3">
                    Welcome, <?php echo htmlspecialchars($_SESSION['admin_username']); ?>
                </span>
                <a href="logout.php" class="btn btn-outline-light btn-sm">Logout</a>
            </div>
        </div>
    </nav>

    <div class="container-fluid">
        <div class="row">
            <nav class="col-md-3 col-lg-2 d-md-block sidebar">
                <div class="sidebar-sticky">
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link" href="dashboard.php">
                                <i class="bi bi-speedometer2"></i> Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link active" href="send_message.php">
                                <i class="bi bi-send"></i> Send Message
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="users.php">
                                <i class="bi bi-people"></i> Users
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="categories.php">
                                <i class="bi bi-folder"></i> Categories
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="messages.php">
                                <i class="bi bi-chat-dots"></i> Messages
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="settings.php">
                                <i class="bi bi-gear"></i> Settings
                            </a>
                        </li>
                    </ul>
                </div>
            </nav>

            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 main">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Send Message</h1>
                </div>

                <?php if ($success): ?>
                    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
                <?php endif; ?>

                <div class="row">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">Compose Message</h5>
                            </div>
                            <div class="card-body">
                                <form action="" method="POST" enctype="multipart/form-data" id="messageForm">
                                    <div class="mb-3">
                                        <label class="form-label">Title</label>
                                        <input type="text" name="title" class="form-control" required>
                                    </div>

                                    <div class="mb-3">
                                        <label class="form-label">Description</label>
                                        <textarea name="description" class="form-control" rows="4" required></textarea>
                                    </div>

                                    <div class="mb-3">
                                        <label class="form-label">Image (Optional)</label>
                                        <input type="file" name="image" class="form-control" accept="image/*">
                                    </div>

                                    <div class="mb-3">
                                        <label class="form-label">Button Text (Optional)</label>
                                        <input type="text" name="button_text" class="form-control">
                                    </div>

                                    <div class="mb-3">
                                        <label class="form-label">Button URL (Optional)</label>
                                        <input type="url" name="button_url" class="form-control">
                                    </div>

                                    <div class="mb-3">
                                        <label class="form-label">Send To</label>
                                        <select name="category_id" class="form-control">
                                            <option value="">All Users</option>
                                            <?php foreach ($categories as $category): ?>
                                                <option value="<?php echo $category['id']; ?>">
                                                    <?php echo htmlspecialchars($category['name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>

                                    <div class="d-grid">
                                        <button type="submit" class="btn btn-primary">
                                            <i class="bi bi-send"></i> Send Message
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>

                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">Message Preview</h5>
                            </div>
                            <div class="card-body">
                                <div id="message-preview">
                                    <div id="preview-image-container" style="display: none;">
                                        <img src="" class="preview-image" id="preview-image">
                                    </div>
                                    <h4 id="preview-title">Message Title</h4>
                                    <p id="preview-description">Message description will appear here.</p>
                                    <div id="preview-button-container" style="display: none;">
                                        <a href="#" class="preview-button" id="preview-button">Button Text</a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Live preview functionality
        document.getElementById('messageForm').addEventListener('input', function(e) {
            if (e.target.name === 'title') {
                document.getElementById('preview-title').textContent = e.target.value || 'Message Title';
            }
            else if (e.target.name === 'description') {
                document.getElementById('preview-description').textContent = e.target.value || 'Message description will appear here.';
            }
            else if (e.target.name === 'button_text') {
                const buttonContainer = document.getElementById('preview-button-container');
                const button = document.getElementById('preview-button');
                if (e.target.value) {
                    buttonContainer.style.display = 'block';
                    button.textContent = e.target.value;
                } else {
                    buttonContainer.style.display = 'none';
                }
            }
            else if (e.target.name === 'button_url') {
                document.getElementById('preview-button').href = e.target.value || '#';
            }
        });

        // Image preview
        document.querySelector('input[name="image"]').addEventListener('change', function(e) {
            const container = document.getElementById('preview-image-container');
            const img = document.getElementById('preview-image');
            
            if (this.files && this.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    container.style.display = 'block';
                    img.src = e.target.result;
                };
                reader.readAsDataURL(this.files[0]);
            } else {
                container.style.display = 'none';
                img.src = '';
            }
        });
    </script>
</body>
</html>