<?php
/**
 * Telegram Bot Web Admin Panel - Message Logs
 */

require_once '../config.php';
require_once '../bot_functions.php';
require_once 'auth_check.php';

$pdo = getDBConnection();

// Get filter parameters
$message_filter = $_GET['message'] ?? '';
$status_filter = $_GET['status'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// Build query
$where_conditions = [];
$params = [];

if ($message_filter) {
    $where_conditions[] = "m.id = ?";
    $params[] = $message_filter;
}

if ($status_filter) {
    $where_conditions[] = "ml.status = ?";
    $params[] = $status_filter;
}

if ($date_from) {
    $where_conditions[] = "DATE(ml.sent_at) >= ?";
    $params[] = $date_from;
}

if ($date_to) {
    $where_conditions[] = "DATE(ml.sent_at) <= ?";
    $params[] = $date_to;
}

$where_clause = $where_conditions ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get message logs
$sql = "
    SELECT ml.*, m.title, u.username, u.first_name, u.last_name, c.name as category_name
    FROM message_logs ml
    LEFT JOIN messages m ON ml.message_id = m.id
    LEFT JOIN users u ON ml.user_id = u.id
    LEFT JOIN categories c ON u.category_id = c.id
    $where_clause
    ORDER BY ml.sent_at DESC
    LIMIT 1000
";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$logs = $stmt->fetchAll();

// Get messages for filter dropdown
$messages = $pdo->query("SELECT id, title FROM messages ORDER BY created_at DESC LIMIT 50")->fetchAll();

// Get statistics
$stats = [
    'total_sent' => 0,
    'total_failed' => 0,
    'total_blocked' => 0
];

if (empty($where_conditions)) {
    $stats_query = $pdo->query("
        SELECT 
            status,
            COUNT(*) as count
        FROM message_logs 
        GROUP BY status
    ");
    
    foreach ($stats_query->fetchAll() as $stat) {
        $stats['total_' . $stat['status']] = $stat['count'];
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Message Logs - Bot Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">📱 Bot Admin</a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="dashboard.php">Dashboard</a>
                <a class="nav-link" href="users.php">Users</a>
                <a class="nav-link" href="send_message.php">Send Message</a>
                <a class="nav-link" href="categories.php">Categories</a>
                <a class="nav-link active" href="message_logs.php">Logs</a>
                <a class="nav-link" href="logout.php">Logout</a>
            </div>
        </div>
    </nav>

    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="fas fa-list"></i> Message Delivery Logs</h2>
                </div>

                <!-- Statistics Cards -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card bg-success text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4><?php echo number_format($stats['total_sent']); ?></h4>
                                        <p class="mb-0">Sent Successfully</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-check-circle fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-danger text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4><?php echo number_format($stats['total_failed']); ?></h4>
                                        <p class="mb-0">Failed</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-times-circle fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-warning text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4><?php echo number_format($stats['total_blocked']); ?></h4>
                                        <p class="mb-0">Blocked</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-ban fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-info text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4><?php echo number_format(count($logs)); ?></h4>
                                        <p class="mb-0">Filtered Results</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-filter fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-3">
                                <label class="form-label">Message</label>
                                <select name="message" class="form-select">
                                    <option value="">All Messages</option>
                                    <?php foreach ($messages as $message): ?>
                                        <option value="<?php echo $message['id']; ?>" 
                                                <?php echo $message_filter == $message['id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars(substr($message['title'], 0, 50)); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Status</label>
                                <select name="status" class="form-select">
                                    <option value="">All Status</option>
                                    <option value="sent" <?php echo $status_filter === 'sent' ? 'selected' : ''; ?>>Sent</option>
                                    <option value="failed" <?php echo $status_filter === 'failed' ? 'selected' : ''; ?>>Failed</option>
                                    <option value="blocked" <?php echo $status_filter === 'blocked' ? 'selected' : ''; ?>>Blocked</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Date From</label>
                                <input type="date" name="date_from" class="form-control" 
                                       value="<?php echo htmlspecialchars($date_from); ?>">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Date To</label>
                                <input type="date" name="date_to" class="form-control" 
                                       value="<?php echo htmlspecialchars($date_to); ?>">
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">&nbsp;</label>
                                <div>
                                    <button type="submit" class="btn btn-primary">Filter</button>
                                    <a href="message_logs.php" class="btn btn-secondary">Clear</a>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Logs Table -->
                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-sm">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Message</th>
                                        <th>User</th>
                                        <th>Category</th>
                                        <th>Status</th>
                                        <th>Sent At</th>
                                        <th>Error</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($logs)): ?>
                                        <tr>
                                            <td colspan="7" class="text-center">No logs found</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($logs as $log): ?>
                                            <tr>
                                                <td><?php echo $log['id']; ?></td>
                                                <td>
                                                    <small>
                                                        <strong>ID:</strong> <?php echo $log['message_id']; ?><br>
                                                        <?php echo htmlspecialchars(substr($log['title'] ?: 'N/A', 0, 30)); ?>
                                                    </small>
                                                </td>
                                                <td>
                                                    <small>
                                                        <?php echo htmlspecialchars($log['first_name'] . ' ' . $log['last_name']); ?><br>
                                                        @<?php echo htmlspecialchars($log['username'] ?: 'N/A'); ?><br>
                                                        <span class="text-muted"><?php echo $log['telegram_id']; ?></span>
                                                    </small>
                                                </td>
                                                <td>
                                                    <small><?php echo htmlspecialchars($log['category_name'] ?: 'N/A'); ?></small>
                                                </td>
                                                <td>
                                                    <?php
                                                    $badge_class = 'secondary';
                                                    switch ($log['status']) {
                                                        case 'sent':
                                                            $badge_class = 'success';
                                                            break;
                                                        case 'failed':
                                                            $badge_class = 'danger';
                                                            break;
                                                        case 'blocked':
                                                            $badge_class = 'warning';
                                                            break;
                                                    }
                                                    ?>
                                                    <span class="badge bg-<?php echo $badge_class; ?>">
                                                        <?php echo ucfirst($log['status']); ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <small><?php echo date('Y-m-d H:i:s', strtotime($log['sent_at'])); ?></small>
                                                </td>
                                                <td>
                                                    <?php if ($log['error_message']): ?>
                                                        <small class="text-danger" title="<?php echo htmlspecialchars($log['error_message']); ?>">
                                                            <?php echo htmlspecialchars(substr($log['error_message'], 0, 30)); ?>...
                                                        </small>
                                                    <?php else: ?>
                                                        <small class="text-muted">-</small>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <div class="mt-3">
                            <small class="text-muted">
                                Showing <?php echo count($logs); ?> logs 
                                <?php if (count($logs) >= 1000): ?>(limited to 1000 most recent)<?php endif; ?>
                            </small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
