-- Telegram Message Sender Bot Database Structure
-- Run this SQL script to create the required database tables

CREATE DATABASE IF NOT EXISTS telegram_bot CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE telegram_bot;

-- Categories table
CREATE TABLE categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL UNIQUE,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Users table
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    telegram_id BIGINT NOT NULL UNIQUE,
    username VARCHAR(100),
    first_name VARCHAR(100),
    last_name VARCHAR(100),
    category_id INT DEFAULT 1,
    is_active BOOLEAN DEFAULT TRUE,
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_activity TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE SET NULL
);

-- Messages table (for storing sent messages and templates)
CREATE TABLE messages (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    image_url VARCHAR(500),
    button_text VARCHAR(100),
    button_url VARCHAR(500),
    category_id INT,
    sent_to_all BOOLEAN DEFAULT FALSE,
    sent_count INT DEFAULT 0,
    created_by VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE SET NULL
);

-- Admins table (for web panel authentication)
CREATE TABLE admins (
    id INT AUTO_INCREMENT PRIMARY KEY,
    telegram_id BIGINT UNIQUE,
    username VARCHAR(100) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_login TIMESTAMP NULL
);

-- Message logs table (track individual message deliveries)
CREATE TABLE message_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    message_id INT,
    user_id INT,
    telegram_id BIGINT,
    status ENUM('sent', 'failed', 'blocked') DEFAULT 'sent',
    error_message TEXT,
    sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (message_id) REFERENCES messages(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Bot sessions table (for managing admin bot conversations)
CREATE TABLE bot_sessions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    telegram_id BIGINT NOT NULL,
    session_data JSON,
    expires_at TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_session (telegram_id)
);

-- Insert default categories
INSERT INTO categories (name, description) VALUES 
('General', 'Default category for all users'),
('Students', 'Student users'),
('Clients', 'Client users'),
('Members', 'Premium members'),
('VIP', 'VIP users');

-- Insert default admin (update with your credentials)
INSERT INTO admins (username, password, telegram_id) VALUES 
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 123456789);
-- Default password is 'password' - CHANGE THIS!

-- Create indexes for better performance
CREATE INDEX idx_users_telegram_id ON users(telegram_id);
CREATE INDEX idx_users_category ON users(category_id);
CREATE INDEX idx_messages_category ON messages(category_id);
CREATE INDEX idx_message_logs_message ON message_logs(message_id);
CREATE INDEX idx_message_logs_user ON message_logs(user_id);
CREATE INDEX idx_bot_sessions_telegram ON bot_sessions(telegram_id);

-- Create views for easier data access
CREATE VIEW user_stats AS
SELECT 
    c.name as category_name,
    COUNT(u.id) as user_count,
    COUNT(CASE WHEN u.is_active = 1 THEN 1 END) as active_users
FROM categories c
LEFT JOIN users u ON c.id = u.category_id
GROUP BY c.id, c.name;

CREATE VIEW message_stats AS
SELECT 
    m.id,
    m.title,
    m.created_at,
    m.sent_count,
    COUNT(ml.id) as delivery_count,
    COUNT(CASE WHEN ml.status = 'sent' THEN 1 END) as successful_deliveries,
    COUNT(CASE WHEN ml.status = 'failed' THEN 1 END) as failed_deliveries
FROM messages m
LEFT JOIN message_logs ml ON m.id = ml.message_id
GROUP BY m.id;
