<?php
/**
 * Cleanup Script
 * Removes test files while preserving project files
 */

// List of test files that can be safely removed
$test_files = [
    __DIR__ . '/test.php',
    __DIR__ . '/test_no_db.php',
    __DIR__ . '/init_db.php',
    __DIR__ . '/bot.db',      // SQLite test database
    __DIR__ . '/test.log',    // Any test log files
];

// List of project files that should never be deleted
$protected_files = [
    __DIR__ . '/bot.php',
    __DIR__ . '/bot_functions.php',
    __DIR__ . '/config.php',
    __DIR__ . '/db.sql',
    __DIR__ . '/schema_sqlite.sql',
    __DIR__ . '/setWebhook.php',
    __DIR__ . '/web-admin/index.php',
    __DIR__ . '/web-admin/dashboard.php',
    __DIR__ . '/web-admin/send_message.php',
    __DIR__ . '/web-admin/auth_check.php',
    __DIR__ . '/web-admin/logout.php',
    __DIR__ . '/web-admin/settings.php',
];

echo "Starting cleanup process...\n\n";

// Function to safely remove a file
function safeRemove($file) {
    global $protected_files;
    
    // Normalize path for comparison
    $normalized_file = str_replace('\\', '/', $file);
    $normalized_protected = array_map(function($f) {
        return str_replace('\\', '/', $f);
    }, $protected_files);
    
    // Check if file is protected
    if (in_array($normalized_file, $normalized_protected)) {
        echo "⚠️ Protected file, skipping: " . basename($file) . "\n";
        return false;
    }
    
    if (file_exists($file)) {
        if (unlink($file)) {
            echo "✅ Removed: " . basename($file) . "\n";
            return true;
        } else {
            echo "❌ Failed to remove: " . basename($file) . "\n";
            return false;
        }
    } else {
        echo "ℹ️ File not found: " . basename($file) . "\n";
        return false;
    }
}

// Clean test files
echo "Removing test files:\n";
echo "-------------------\n";
foreach ($test_files as $file) {
    safeRemove($file);
}

// Clean test uploads
$upload_dir = __DIR__ . '/uploads/';
if (is_dir($upload_dir)) {
    echo "\nCleaning test uploads:\n";
    echo "-------------------\n";
    
    $files = glob($upload_dir . 'test_*');
    foreach ($files as $file) {
        safeRemove($file);
    }
    
    // Remove empty uploads directory if it was only used for tests
    if (is_dir($upload_dir) && count(glob($upload_dir . '*')) === 0) {
        if (rmdir($upload_dir)) {
            echo "✅ Removed empty uploads directory\n";
        }
    }
}

echo "\nCleanup Summary:\n";
echo "---------------\n";
echo "Protected files: " . count($protected_files) . "\n";
echo "Test files checked: " . count($test_files) . "\n";

if (file_exists($upload_dir)) {
    echo "Remaining upload files: " . count(glob($upload_dir . '*')) . "\n";
}

echo "\nNote: Project files are preserved and ready for production use.\n";