<?php
/**
 * Telegram Bot Functions Library
 * Contains all modular functions for bot operations
 */

require_once 'config.php';

/**
 * Send HTTP request to Telegram API
 */
function telegramRequest($method, $parameters = []) {
    $url = TELEGRAM_API . $method;
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $parameters);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code !== 200) {
        error_log("Telegram API Error: HTTP $http_code - $response");
        return false;
    }
    
    return json_decode($response, true);
}

/**
 * Send text message
 */
function sendMessage($chat_id, $text, $reply_markup = null, $parse_mode = 'HTML') {
    $parameters = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => $parse_mode
    ];
    
    if ($reply_markup) {
        $parameters['reply_markup'] = json_encode($reply_markup);
    }
    
    return telegramRequest('sendMessage', $parameters);
}

/**
 * Send photo with caption
 */
function sendPhoto($chat_id, $photo, $caption = '', $reply_markup = null) {
    $parameters = [
        'chat_id' => $chat_id,
        'photo' => $photo,
        'caption' => $caption,
        'parse_mode' => 'HTML'
    ];
    
    if ($reply_markup) {
        $parameters['reply_markup'] = json_encode($reply_markup);
    }
    
    return telegramRequest('sendPhoto', $parameters);
}

/**
 * Edit message text
 */
function editMessageText($chat_id, $message_id, $text, $reply_markup = null) {
    $parameters = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    
    if ($reply_markup) {
        $parameters['reply_markup'] = json_encode($reply_markup);
    }
    
    return telegramRequest('editMessageText', $parameters);
}

/**
 * Answer callback query
 */
function answerCallbackQuery($callback_query_id, $text = '', $show_alert = false) {
    return telegramRequest('answerCallbackQuery', [
        'callback_query_id' => $callback_query_id,
        'text' => $text,
        'show_alert' => $show_alert
    ]);
}

/**
 * Register or update user in database
 */
function registerUser($telegram_id, $username, $first_name, $last_name) {
    try {
        $pdo = getDBConnection();
        
        $stmt = $pdo->prepare("
            INSERT INTO users (telegram_id, username, first_name, last_name, category_id) 
            VALUES (?, ?, ?, ?, 1)
            ON DUPLICATE KEY UPDATE 
                username = VALUES(username),
                first_name = VALUES(first_name),
                last_name = VALUES(last_name),
                last_activity = CURRENT_TIMESTAMP
        ");
        
        return $stmt->execute([$telegram_id, $username, $first_name, $last_name]);
    } catch (Exception $e) {
        error_log("Error registering user: " . $e->getMessage());
        return false;
    }
}

/**
 * Get user from database
 */
function getUser($telegram_id) {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->prepare("
            SELECT u.*, c.name as category_name 
            FROM users u 
            LEFT JOIN categories c ON u.category_id = c.id 
            WHERE u.telegram_id = ?
        ");
        $stmt->execute([$telegram_id]);
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log("Error getting user: " . $e->getMessage());
        return false;
    }
}

/**
 * Get all categories
 */
function getCategories() {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->query("SELECT * FROM categories ORDER BY name");
        return $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Error getting categories: " . $e->getMessage());
        return [];
    }
}

/**
 * Get users by category
 */
function getUsersByCategory($category_id = null) {
    try {
        $pdo = getDBConnection();
        
        if ($category_id) {
            $stmt = $pdo->prepare("
                SELECT * FROM users 
                WHERE category_id = ? AND is_active = 1 
                ORDER BY joined_at DESC
            ");
            $stmt->execute([$category_id]);
        } else {
            $stmt = $pdo->query("
                SELECT * FROM users 
                WHERE is_active = 1 
                ORDER BY joined_at DESC
            ");
        }
        
        return $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Error getting users by category: " . $e->getMessage());
        return [];
    }
}

/**
 * Send message to category
 */
function sendToCategory($category_id, $message_data) {
    $users = getUsersByCategory($category_id);
    $sent_count = 0;
    $failed_count = 0;
    
    foreach ($users as $user) {
        $success = sendFormattedMessage($user['telegram_id'], $message_data);
        if ($success) {
            $sent_count++;
            logMessageDelivery($message_data['message_id'], $user['id'], $user['telegram_id'], 'sent');
        } else {
            $failed_count++;
            logMessageDelivery($message_data['message_id'], $user['id'], $user['telegram_id'], 'failed');
        }
        
        // Small delay to avoid rate limiting
        usleep(100000); // 0.1 second
    }
    
    return ['sent' => $sent_count, 'failed' => $failed_count];
}

/**
 * Send message to all users
 */
function sendToAll($message_data) {
    $users = getUsersByCategory();
    $sent_count = 0;
    $failed_count = 0;
    
    foreach ($users as $user) {
        $success = sendFormattedMessage($user['telegram_id'], $message_data);
        if ($success) {
            $sent_count++;
            logMessageDelivery($message_data['message_id'], $user['id'], $user['telegram_id'], 'sent');
        } else {
            $failed_count++;
            logMessageDelivery($message_data['message_id'], $user['id'], $user['telegram_id'], 'failed');
        }
        
        // Small delay to avoid rate limiting
        usleep(100000); // 0.1 second
    }
    
    return ['sent' => $sent_count, 'failed' => $failed_count];
}

/**
 * Send formatted message with image, title, description and button
 */
function sendFormattedMessage($chat_id, $message_data) {
    $caption = '';
    
    if (!empty($message_data['title'])) {
        $caption .= "<b>" . htmlspecialchars($message_data['title']) . "</b>\n\n";
    }
    
    if (!empty($message_data['description'])) {
        $caption .= htmlspecialchars($message_data['description']);
    }
    
    $reply_markup = null;
    if (!empty($message_data['button_text']) && !empty($message_data['button_url'])) {
        $reply_markup = [
            'inline_keyboard' => [[
                [
                    'text' => $message_data['button_text'],
                    'url' => $message_data['button_url']
                ]
            ]]
        ];
    }
    
    if (!empty($message_data['image_url'])) {
        return sendPhoto($chat_id, $message_data['image_url'], $caption, $reply_markup);
    } else {
        return sendMessage($chat_id, $caption, $reply_markup);
    }
}

/**
 * Save message to database
 */
function saveMessage($title, $description, $image_url, $button_text, $button_url, $category_id, $created_by) {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->prepare("
            INSERT INTO messages (title, description, image_url, button_text, button_url, category_id, created_by) 
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([$title, $description, $image_url, $button_text, $button_url, $category_id, $created_by]);
        return $pdo->lastInsertId();
    } catch (Exception $e) {
        error_log("Error saving message: " . $e->getMessage());
        return false;
    }
}

/**
 * Log message delivery
 */
function logMessageDelivery($message_id, $user_id, $telegram_id, $status, $error_message = null) {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->prepare("
            INSERT INTO message_logs (message_id, user_id, telegram_id, status, error_message) 
            VALUES (?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([$message_id, $user_id, $telegram_id, $status, $error_message]);
    } catch (Exception $e) {
        error_log("Error logging message delivery: " . $e->getMessage());
        return false;
    }
}

/**
 * Get or create bot session
 */
function getBotSession($telegram_id) {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->prepare("
            SELECT session_data FROM bot_sessions 
            WHERE telegram_id = ? AND expires_at > NOW()
        ");
        $stmt->execute([$telegram_id]);
        $session = $stmt->fetch();
        
        if ($session) {
            return json_decode($session['session_data'], true);
        }
        
        return [];
    } catch (Exception $e) {
        error_log("Error getting bot session: " . $e->getMessage());
        return [];
    }
}

/**
 * Save bot session
 */
function saveBotSession($telegram_id, $session_data) {
    try {
        $pdo = getDBConnection();
        $expires_at = date('Y-m-d H:i:s', strtotime('+1 hour'));
        
        $stmt = $pdo->prepare("
            INSERT INTO bot_sessions (telegram_id, session_data, expires_at) 
            VALUES (?, ?, ?)
            ON DUPLICATE KEY UPDATE 
                session_data = VALUES(session_data),
                expires_at = VALUES(expires_at),
                updated_at = CURRENT_TIMESTAMP
        ");
        
        return $stmt->execute([$telegram_id, json_encode($session_data), $expires_at]);
    } catch (Exception $e) {
        error_log("Error saving bot session: " . $e->getMessage());
        return false;
    }
}

/**
 * Clear bot session
 */
function clearBotSession($telegram_id) {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->prepare("DELETE FROM bot_sessions WHERE telegram_id = ?");
        return $stmt->execute([$telegram_id]);
    } catch (Exception $e) {
        error_log("Error clearing bot session: " . $e->getMessage());
        return false;
    }
}

/**
 * Get user statistics
 */
function getUserStats() {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->query("SELECT * FROM user_stats ORDER BY category_name");
        return $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Error getting user stats: " . $e->getMessage());
        return [];
    }
}
?>
